"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isRealTimeTask = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getResultAggregationQuery = exports.getLatestTaskForDetectorQuery = exports.getLatestDetectorTasksQuery = exports.getFinalDetectorStates = exports.getFiltersFromEntityList = exports.getErrorMessage = exports.getDetectorsWithJob = exports.getDetectorTasks = exports.getDetectorResults = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertPreviewInputKeysToSnakeCase = exports.convertDetectorKeysToSnakeCase = exports.convertDetectorKeysToCamelCase = exports.appendTaskInfo = exports.anomalyResultMapper = void 0;
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
var _constants = require("../../utils/constants");
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const convertDetectorKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,
    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};
exports.convertDetectorKeysToSnakeCase = convertDetectorKeysToSnakeCase;
const convertPreviewInputKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['detector']) // Exclude the detector,
    }, _helpers.toSnake),
    detector: convertDetectorKeysToSnakeCase((0, _lodash.get)(payload, 'detector', {}))
  };
};
exports.convertPreviewInputKeysToSnakeCase = convertPreviewInputKeysToSnakeCase;
const convertDetectorKeysToCamelCase = response => {
  let camelCaseResponse = {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'adJob', 'historicalTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };
  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'historicalTask', {}))) {
    camelCaseResponse = {
      ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'historicalTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'historicalTask', {})),
      taskProgress: (0, _lodash.get)(response, 'historicalTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'historicalTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }
  return camelCaseResponse;
};

// Converts the static detector fields into camelcase. Ignores any job or task-related fields
exports.convertDetectorKeysToCamelCase = convertDetectorKeysToCamelCase;
const convertStaticFieldsToCamelCase = response => {
  return {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'anomaly_detector_job', 'anomaly_detection_task', 'realtime_detection_task', 'historical_analysis_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {})
  };
};

// Converts the task-related detector fields into camelcase
exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;
const convertTaskAndJobFieldsToCamelCase = (realtimeTask, historicalTask, detectorJob) => {
  let response = {};

  // Populate AD job fields
  response = {
    ...response,
    enabled: (0, _lodash.get)(detectorJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(detectorJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(detectorJob, 'disabled_time')
  };

  // Populate RT-task-related fields
  response = realtimeTask !== undefined ? {
    ...response,
    curState: getTaskState(realtimeTask),
    stateError: processTaskError((0, _lodash.get)(realtimeTask, 'error', '')),
    initProgress: getTaskInitProgress(realtimeTask)
  } : {
    ...response,
    curState: (0, _lodash.get)(detectorJob, 'enabled', false) ? _constants.DETECTOR_STATE.RUNNING : _constants.DETECTOR_STATE.DISABLED
  };

  // Detection date range field is stored under the 'detector' field in legacy historical tasks.
  // To handle this, need to add a check to fetch the date range from the correct place
  const isLegacyHistorical = (0, _lodash.get)(historicalTask, 'detection_date_range') === undefined && (0, _lodash.get)(historicalTask, 'detector.detection_date_range') !== undefined;
  const legacyDateRangePrefix = isLegacyHistorical ? 'detector.' : '';

  // Populate historical-task-related fields
  response = {
    ...response,
    taskId: (0, _lodash.get)(historicalTask, 'id'),
    taskState: getTaskState(historicalTask),
    taskProgress: (0, _lodash.get)(historicalTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(historicalTask, 'error', '')),
    detectionDateRange: {
      startTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.start_time`),
      endTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.end_time`)
    }
  };
  if ((0, _lodash.isEmpty)(historicalTask)) {
    //@ts-ignore
    delete response.detectionDateRange;
  }
  return response;
};
exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;
const getResultAggregationQuery = (detectors, queryParams) => {
  const aggregationSort = {
    totalAnomalies: 'total_anomalies_in_24hr',
    latestAnomalyTime: 'latest_anomaly_time'
  };
  let aggsSortingOrder = {};
  if (aggregationSort[queryParams.sortField]) {
    aggsSortingOrder = {
      order: {
        [aggregationSort[queryParams.sortField]]: queryParams.sortDirection
      }
    };
  }
  return {
    size: 0,
    query: {
      bool: {
        must: [{
          terms: {
            detector_id: detectors
          }
        }, {
          range: {
            anomaly_grade: {
              gt: 0
            }
          }
        }],
        must_not: {
          exists: {
            field: 'task_id'
          }
        }
      }
    },
    aggs: {
      unique_detectors: {
        terms: {
          field: 'detector_id',
          size: queryParams.from + queryParams.size,
          ...aggsSortingOrder
        },
        aggs: {
          total_anomalies_in_24hr: {
            filter: {
              range: {
                data_start_time: {
                  gte: 'now-24h',
                  lte: 'now'
                }
              }
            }
          },
          latest_anomaly_time: {
            max: {
              field: 'data_start_time'
            }
          }
        }
      }
    }
  };
};
exports.getResultAggregationQuery = getResultAggregationQuery;
const anomalyResultMapper = anomalyResults => {
  let resultData = {
    anomalies: [],
    featureData: {}
  };
  if (anomalyResults.length === 0) return resultData;
  //initialize features list.
  const firstAnomaly = anomalyResults[0];
  Object.values(firstAnomaly.featureData).forEach(feature => {
    resultData.featureData[feature.featureId] = [];
  });
  anomalyResults.forEach(({
    featureData,
    ...rest
  }) => {
    const {
      dataStartTime,
      dataEndTime,
      ...others
    } = rest;
    resultData.anomalies.push({
      ...others,
      anomalyGrade: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.anomalyGrade).toFixed(2) : 0,
      confidence: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.confidence).toFixed(2) : 0,
      startTime: rest.dataStartTime,
      endTime: rest.dataEndTime,
      plotTime: rest.dataEndTime,
      ...(rest.entity !== undefined ? {
        entity: rest.entity
      } : {})
    });
    featureData.forEach(feature => {
      resultData.featureData[feature.featureId].push({
        startTime: rest.dataStartTime,
        endTime: rest.dataEndTime,
        plotTime: rest.dataEndTime,
        data: feature.data
      });
    });
  });
  return resultData;
};
exports.anomalyResultMapper = anomalyResultMapper;
const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }
  return undefined;
};
exports.getTaskInitProgress = getTaskInitProgress;
const getFinalDetectorStates = (detectorStateResponses, finalDetectors) => {
  let finalDetectorStates = (0, _lodash.cloneDeep)(detectorStateResponses);
  finalDetectorStates.forEach(detectorState => {
    //@ts-ignore
    detectorState.state = _constants.DETECTOR_STATE[detectorState.state];
  });

  // check if there was any failures / detectors that are unable to start
  finalDetectorStates.forEach((detectorState, i) => {
    /*
      If the error starts with 'Stopped detector', then an EndRunException was thrown.
      All EndRunExceptions are related to initialization failures except for the
      unknown prediction error which contains the message "We might have bugs".
    */
    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && detectorState.error !== undefined && detectorState.error.includes('Stopped detector')) {
      detectorState.state = detectorState.error.includes('We might have bugs') ? _constants.DETECTOR_STATE.UNEXPECTED_FAILURE : _constants.DETECTOR_STATE.INIT_FAILURE;
    }

    /*
      If a detector is disabled and has no features, set to
      a feature required state
    */
    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && finalDetectors[i].featureAttributes.length === 0) {
      detectorState.state = _constants.DETECTOR_STATE.FEATURE_REQUIRED;
    }
  });
  return finalDetectorStates;
};
exports.getFinalDetectorStates = getFinalDetectorStates;
const getDetectorsWithJob = detectorsWithJobResponses => {
  const finalDetectorsWithJobResponses = (0, _lodash.cloneDeep)(detectorsWithJobResponses);
  const resultDetectorWithJobs = [];
  finalDetectorsWithJobResponses.forEach(detectorWithJobResponse => {
    const resp = {
      ...detectorWithJobResponse.anomaly_detector,
      id: detectorWithJobResponse._id,
      primaryTerm: detectorWithJobResponse._primary_term,
      seqNo: detectorWithJobResponse._seq_no,
      adJob: {
        ...detectorWithJobResponse.anomaly_detector_job
      },
      historicalTask: {
        ...detectorWithJobResponse.anomaly_detection_task
      }
    };
    resultDetectorWithJobs.push(convertDetectorKeysToCamelCase(resp));
  });
  return resultDetectorWithJobs;
};
exports.getDetectorsWithJob = getDetectorsWithJob;
const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};
exports.isIndexNotFoundError = isIndexNotFoundError;
const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};
exports.getErrorMessage = getErrorMessage;
const getDetectorTasks = detectorTaskResponses => {
  const detectorToTaskMap = {};
  detectorTaskResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, '_id', '');
    const detectorTask = (0, _lodash.get)(response, 'anomaly_detection_task', null);
    if (detectorTask !== null) {
      detectorToTaskMap[detectorId] = detectorTask;
    }
  });
  return detectorToTaskMap;
};
exports.getDetectorTasks = getDetectorTasks;
const getDetectorResults = detectorResultsResponses => {
  const detectorToResultsMap = {};
  detectorResultsResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, 'hits.hits.0._source.detector_id', null);
    if (detectorId !== null) {
      detectorToResultsMap[detectorId] = response;
    }
  });
  return detectorToResultsMap;
};

// Append task-related info - task state & anomaly results of the task.
// If there is no related task info for a detector: set to default values of DISABLED state and 0 anomalies
exports.getDetectorResults = getDetectorResults;
const appendTaskInfo = (detectorMap, detectorToTaskMap, detectorToResultsMap) => {
  const detectorMapWithTaskInfo = {};
  const detectorsWithTasks = Object.keys(detectorToTaskMap);
  Object.keys(detectorMap).forEach((detectorId, index) => {
    if (!detectorsWithTasks.includes(detectorId)) {
      detectorMapWithTaskInfo[detectorId] = {
        ...detectorMap[detectorId],
        curState: _constants.DETECTOR_STATE.DISABLED,
        totalAnomalies: 0
      };
    } else {
      const task = detectorToTaskMap[detectorId];
      const state = getTaskState(task);
      const totalAnomalies = (0, _lodash.get)(detectorToResultsMap[detectorId], 'hits.total.value', 0);
      detectorMapWithTaskInfo[detectorId] = {
        ...detectorMap[detectorId],
        curState: state,
        totalAnomalies: totalAnomalies
      };
    }
  });
  return detectorMapWithTaskInfo;
};

// Following checks/transformations need to be made here:
// - set to DISABLED if there is no existing task for this detector
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INIT if the task is in a CREATED state
// - set to DISABLED if the task is in a STOPPED state
exports.appendTaskInfo = appendTaskInfo;
const getTaskState = task => {
  const state = (0, _lodash.get)(task, 'state', 'DISABLED');
  const errorMessage = processTaskError((0, _lodash.get)(task, 'error', ''));
  const updatedState = state === 'FAILED' && errorMessage.includes(_constants.STACK_TRACE_PATTERN) ? 'UNEXPECTED_FAILURE' : state === 'CREATED' ? 'INIT' : state === 'STOPPED' ? 'DISABLED' : state;
  //@ts-ignore
  return _constants.DETECTOR_STATE[updatedState];
};
exports.getTaskState = getTaskState;
const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
};

// Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.processTaskError = processTaskError;
const getLatestDetectorTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      detectors: {
        terms: {
          field: 'detector_id',
          size: _constants.MAX_DETECTORS
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};
exports.getLatestDetectorTasksQuery = getLatestDetectorTasksQuery;
const isRealTimeTask = task => {
  return (0, _lodash.get)(task, 'task_type', '').includes(_constants.REALTIME_TASK_TYPE_PREFIX);
};
exports.isRealTimeTask = isRealTimeTask;
const getFiltersFromEntityList = entityListAsObj => {
  let filters = [];
  Object.values(entityListAsObj).forEach(entity => {
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_NAME_PATH_FIELD]: {
              value: entity.name
            }
          }
        }
      }
    });
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_VALUE_PATH_FIELD]: {
              value: entity.value
            }
          }
        }
      }
    });
  });
  return filters;
};

// Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.getFiltersFromEntityList = getFiltersFromEntityList;
const getLatestTaskForDetectorQuery = (detectorId, realtime) => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            detector_id: detectorId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};
exports.getLatestTaskForDetectorQuery = getLatestTaskForDetectorQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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