﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codebuild/CodeBuildRequest.h>
#include <aws/codebuild/CodeBuild_EXPORTS.h>
#include <aws/codebuild/model/TestCaseFilter.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeBuild {
namespace Model {

/**
 */
class DescribeTestCasesRequest : public CodeBuildRequest {
 public:
  AWS_CODEBUILD_API DescribeTestCasesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeTestCases"; }

  AWS_CODEBUILD_API Aws::String SerializePayload() const override;

  AWS_CODEBUILD_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The ARN of the report for which test cases are returned. </p>
   */
  inline const Aws::String& GetReportArn() const { return m_reportArn; }
  inline bool ReportArnHasBeenSet() const { return m_reportArnHasBeenSet; }
  template <typename ReportArnT = Aws::String>
  void SetReportArn(ReportArnT&& value) {
    m_reportArnHasBeenSet = true;
    m_reportArn = std::forward<ReportArnT>(value);
  }
  template <typename ReportArnT = Aws::String>
  DescribeTestCasesRequest& WithReportArn(ReportArnT&& value) {
    SetReportArn(std::forward<ReportArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> During a previous call, the maximum number of items that can be returned is
   * the value specified in <code>maxResults</code>. If there more items in the list,
   * then a unique string called a <i>nextToken</i> is returned. To get the next
   * batch of items in the list, call this operation again, adding the next token to
   * the call. To get all of the items in the list, keep calling this operation with
   * each subsequent next token that is returned, until no more next tokens are
   * returned. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeTestCasesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The maximum number of paginated test cases returned per response. Use
   * <code>nextToken</code> to iterate pages in the list of returned
   * <code>TestCase</code> objects. The default value is 100. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeTestCasesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A <code>TestCaseFilter</code> object used to filter the returned reports.
   * </p>
   */
  inline const TestCaseFilter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = TestCaseFilter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = TestCaseFilter>
  DescribeTestCasesRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_reportArn;
  bool m_reportArnHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  TestCaseFilter m_filter;
  bool m_filterHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeBuild
}  // namespace Aws
